/**
 * \file: libsugc_config.h
 *
 * \brief : Functions to read the configuration
 *
 * \author: Christoph Gellner (cgellner@de.adit-jv.com)
 *
 * \copyright (c) 2017 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 *
 ***********************************************************************/

#ifndef LIBSUGC_CONFIG
#define LIBSUGC_CONFIG

#include <libsugc.h>
#include <sys/capability.h>

/**
 * \defgroup libsugc_config_api libsugc internal API to read config from file
 */

/**
 * \ingroup libsugc_config_api
 * \brief define for the config file
 */
#define LIBSUGC_CONF "/etc/libsugc.conf"

/**
 * \ingroup libsugc_config_api
 * \brief Data structure to store the read configuration
 */
typedef struct libsugc_config_t {
    uid_t uid;              /**< UID of the user specified in the configuration */
    gid_t gid;              /**< GID of the group  specified in the configuration or primary group of user if no group specified */
    cap_t caps;             /**< Configured capabilities or NULL(no capabilities set - clear all capabilities) */
    size_t num_supp_grps;   /**< number of groups in \ref supp_grps */
    gid_t *supp_grps;       /**< supplementary group of the user specified by \ref uid */
} libsugc_config_t;

/**
 * \ingroup libsugc_config_api
 * \brief Initialize the configuration(\ref libsugc_config_t)
 *
 * \param[in] config        \ref libsugc_config_t struct to initialize
 *
 * \return \ref libsugc_error_t
 */
libsugc_error_t libsugc_init_config(libsugc_config_t *config);

/**
 * \ingroup libsugc_config_api
 * \brief Free all memory associated to the configuration(\ref libsugc_config_t)
 *
 * \param[in] config        \ref libsugc_config_t struct to free
 *
 * \return \ref libsugc_error_t
 */
libsugc_error_t libsugc_free_config(libsugc_config_t *config);

/**
 * \ingroup libsugc_config_api
 * \brief Read the configuration from file and store to \ref libsugc_config_t struct
 *
 * \param[in] config_name   name of the entry in config file(\ref LIBSUGC_CONF)
 * \param[out] config       initialized \ref libsugc_config_t struct to fill
 *
 * \return \ref libsugc_error_t
 */
libsugc_error_t libsugc_fill_config(const char *config_name, libsugc_config_t *config);

#endif
